module BTree where

-- This module implements an order 3 B-Tree (see D.E.Knuth: TAOCP.III)

import Data.List(sort,nub,unfoldr,(\\),intersect)
-- Tree a
--      the type for B-trees of items of type a

data (Ord a, Read a, Show a) => 
		Tree a =  Leaf
			| Node2 (Tree a) a (Tree a)
			| Node3 (Tree a) a (Tree a) a (Tree a)
	deriving (Eq,Read,Show)

-- findMax: the the maximum of a non-empty tree
findMax Leaf = Nothing
findMax (Node2 _ a Leaf) = Just a
findMax (Node3 _ _ _ b Leaf) = Just b
findMax (Node2 _ _ r) = findMax r
findMax (Node3 _ _ _ _ r) = findMax r

-- findMin the minimum of a nenempty tree

findMin Leaf = Nothing
findMin (Node2 Leaf a Leaf) = Just a
findMin (Node3 Leaf a Leaf _ Leaf) = Just a
findMin (Node2 l _ _) = findMin l
findMin (Node3 l _ _ _ _) = findMin l

-- search for some key in a tree
search Leaf _ = Nothing
search (Node2 l a r) x
	| x < a = search l x
	| x == a = (Just a)
	| x > a = search r x
search (Node3 l a m b r) x
	| x < a = search l x
	| x == a = (Just a)
	| x < b = search m x
	| x == b = (Just b)
	| x > b = search r x
 
--  TreeSplit a
--    A tag to define whether a node has ben split
--     (Split n) means n is the result of a split
data TreeSplit a = 	  Split (Tree a)
			| NoSplit (Tree a)
	deriving (Eq,Read,Show)

-- insert x in a tree t
insert t x = case (recinsert t x) of
		(Split newt) -> newt
		(NoSplit newt) -> newt

-- recinsert
-- the real insertion procedure
recinsert Leaf  x = Split (Node2 Leaf x Leaf)
recinsert t@(Node2 l a r) x 
	| x < a = case (recinsert l x) of
			(Split (Node2 ln n rn)) -> NoSplit (Node3 ln n rn a r)
			(NoSplit newl) -> NoSplit (Node2 newl a r)
	| x == a = NoSplit t
	| x > a = case (recinsert r x) of
			(Split (Node2 ln n rn)) -> NoSplit (Node3 l a ln n rn)
			(NoSplit newr) -> NoSplit (Node2 l a newr)

recinsert t@(Node3 l a m b r) x
	| x < a = case (recinsert l x) of
			(Split newl) -> Split (Node2 newl a (Node2 m b r))
			(NoSplit newl) -> NoSplit (Node3 newl a m b r)
	| x == a = NoSplit t
	| x < b = case (recinsert  m x) of
			(Split (Node2 lm up rm)) -> Split (Node2 newl up newr)
				where newl = Node2 l a lm
				      newr = Node2 rm b r
			(NoSplit newm) -> NoSplit (Node3 l a newm b r)
	| x == b = NoSplit t
	| x > b = case (recinsert r x) of
			(Split newr) -> Split (Node2 newl b newr)
				where newl = Node2 l a m
			(NoSplit newr) -> NoSplit (Node3 l a m b newr)


-- TreeHole a
--- type used as tag to denote the result of a deletion produced a Hole
data TreeHole a = Hole (Tree a)
		| NoHole (Tree a)

-- Hole merging functions


-- hole merging when parent has 1 item (Node2)
par2LftHoleMrg down a (Node2 lr ar rr) = Hole (Node3 down a lr ar rr)
par2LftHoleMrg down a (Node3 lr ar mr br rr) = NoHole (Node2 (Node2 down a lr) 
						             ar
	 					             (Node2 mr br rr)) 
par2RgtHoleMrg (Node2 ll al rl) a down = Hole (Node3 ll al rl a down)
par2RgtHoleMrg (Node3 ll al ml bl rl) a down= NoHole (Node2 (Node2 ll al ml)
						            bl
						            (Node2 rl a down)) 

-- hole merging when parent has 2 items (Node3)
par3LftHoleMrg down a (Node2 lm am rm) b r = 
		NoHole (Node2 (Node3 down a lm am rm) b r)
par3LftHoleMrg down a (Node3 lm am mm bm rm) b r =
		NoHole (Node3 (Node2 down a lm) am (Node2 mm bm rm) b r)

par3MdlHoleMrg (Node2 ll al rl) a down b r = 	
		NoHole (Node2 (Node3 ll al rl a down) b r)
par3MdlHoleMrg (Node3 ll al ml bl rl) a down b r =
		NoHole (Node3 (Node2 ll al ml) bl (Node2 rl a down) b r)

par3RgtHoleMrg l a (Node2 lm am rm) b down =
		NoHole (Node2 l a (Node3 lm am rm b down))
par3RgtHoleMrg l a (Node3 lm am mm bm rm) b down =
		NoHole (Node3 l a (Node2 lm am mm) bm (Node2 rm b down))


-- delete 
-- given tree t and item x produces a new tree, resulting deleting x from t
			
delete t x = case (recdelete t x) of
		(Hole newt) -> newt
		(NoHole newt) -> newt


recdelete Leaf _ = NoHole Leaf

recdelete t@(Node2 Leaf a Leaf) x 
	| x == a = Hole Leaf
	| otherwise = NoHole t
recdelete t@(Node3 Leaf a Leaf b Leaf) x
	| x == a = NoHole (Node2 Leaf b Leaf)
	| x == b = NoHole (Node2 Leaf a Leaf)
	| otherwise = NoHole t
recdelete t@(Node2 l a r) x
	| x < a = case (recdelete l x) of
			(Hole down) -> (par2LftHoleMrg down a r)
			(NoHole newl) -> NoHole (Node2 newl a r)
	| x == a = case (recdelMax l) of
 			(max,Hole down) -> (par2LftHoleMrg down max r)
			(max, NoHole newl) -> NoHole (Node2 newl max r)
	| x > a = case (recdelete  r x) of
			(Hole down) -> par2RgtHoleMrg l a down
			(NoHole newr) -> NoHole (Node2 l a newr)
recdelete t@(Node3 l a m b r) x
	| x < a = case (recdelete l x) of
			(Hole down) -> par3LftHoleMrg down a m b r
			(NoHole newl) -> NoHole (Node3 newl a m b r)
	| x == a = case (recdelMax l) of
			(max, Hole down) -> par3LftHoleMrg down max m b r
			(max, NoHole newl) -> NoHole (Node3 newl max m b r)
	| x < b = case (recdelete m x) of 
			(Hole down) -> par3MdlHoleMrg l a down b r
			(NoHole newm) -> NoHole (Node3 l a newm b r)
	| x == b = case (recdelMax m) of
			(max, Hole down) -> par3MdlHoleMrg l a down max r
			(max, NoHole down) -> NoHole (Node3 l a down max r)
	| x > b = case (recdelete r x) of
			(Hole down) -> par3RgtHoleMrg l a m b down
			(NoHole newr) -> NoHole (Node3 l a m b newr)  
			 

recdelMax t  
	| Just max <- findMax t = (max, recdelete t max)
	| otherwise = error "recdelmax: 2-3 tree empty"
	
-- deleteMax: delete the maximum

deleteMax t  	
	| Just max <- findMax t = delete t max
        | otherwise = t

-- splitMax: return pair with maximum and new tree resulting from excluding maximum
splitMax t 	
	| Just max <- findMax t = Just (max,delete t max)
	| otherwise = Nothing

-- deleteMin: exclude minimum to give new tree
deleteMin t 
	| Just min <- findMin t = delete t min
	| otherwise = t

-- splitMin: return pair with minmum and new tree resulting from minimum exclusion

splitMin t 	
	| Just min <- findMin t = Just (min,delete t min)
	| otherwise = Nothing

-- btreeFoldl: build tree from a list
btreeFoldl :: (Ord a,Read a, Show a) => [a] -> (Tree a)
btreeFoldl = foldl insert Leaf  

-- btreeFoldr: buld tree from list
btreeFoldr :: (Ord a,Read a, Show a) => [a] -> (Tree a)
btreeFoldr = foldr (\ x -> \t -> insert t x) Leaf  

-- btreeUnfoldr: build list from tree (or ordered list of items in tree)
btreeUnfoldr :: (Ord a,Read a,Show a) => Tree a->[a]
btreeUnfoldr = unfoldr splitMin 

-- btreeFlatten: flatten to tree list (or ordered list of items in tree)
--   the same as btreeUnfoldr, but unnecessarily complicated
btreeFlatten :: (Ord a,Read a,Show a) => (Tree a) -> [a]
btreeFlatten Leaf = []
btreeFlatten (Node2 l a r) = (btreeFlatten l)++a:(btreeFlatten r)
btreeFlatten (Node3 l a m b r) = (btreeFlatten l)++
                                 a:((btreeFlatten m)++
                                 b:(btreeFlatten r))

-- test whether b-tree is ordered
btreeOrdered :: (Ord a,Read a,Show a)=> (Tree a) -> Bool
btreeOrdered t = 
	let xs = btreeFlatten t
	in all (\ (x,y) -> (x <= y)) (zip xs (tail xs))

-- test whether b-tree is balanced
balanced t = case (heightBal t) of
		(Just n) -> True
		Nothing -> False

heightBal Leaf = Just 0
heightBal (Node2 l a r) 
	| Just hl <- heightBal l
	, Just hr <- heightBal r
        , hl == hr = Just (hl+1)
        | otherwise = Nothing
heightBal (Node3 l a m b r) 
	| Just hl <- heightBal l
	, Just hr <- heightBal r
	, Just hm <- heightBal m
        , hl == hm
        , hm == hr = Just (hl+1)
        | otherwise = Nothing

btreeOK t = (balanced t) && (btreeOrdered t)

