module Main where

import Point
import Control.Exception
import System.CPUTime

import Accept
import KDtree
import PQ
import Distance

main = readIntPts [] >>= \pts -> let tour = if accept pts
                                            then pqGreedy pts
                                            else error "\nErro PQgreedy.014 -> O Caixeiro Viajante nao consegue entender o mapa (input invalido)"
                                 in putStr (showPoints (tour))

--------------------------------------------------------------------------------
-- Verifica o grau de um ponto

isPresent _ (LeafKD _ present) = present

isPresent (x, y) (Node2KD left v cut right) =
   if coord <= v
   then isPresent (x, y) left
   else isPresent (x, y) right
   where coord = if cut then x else y

--------------------------------------------------------------------------------
-- Constroi um fila de prioridade de arestas a partir de uma KDtree

allEdges [] _ = []

allEdges (p:ps) tree = (nearSearchWE p tree):(allEdges ps tree)

buildPQ [] = LeafPQ

buildPQ (e:es) = insertPQ (buildPQ es) e

--------------------------------------------------------------------------------
-- Um tour pode ter varios pedacos desconectados, cada pedaco do tour
-- e representado como ([Point a], [Point a, Point a])
--                     ([meio], [inicio, fim]) <=> [inicio] ++ [meio] ++ [fim]

--------------------------------------------------------------------------------
-- Dado um tour, cria uma lista com as arestas que faltam para conectar os
-- pedacos do tour

edges e [] = []

edges e [p] = [m13, m14, m23, m24]
              where [p1, p2] = e
                    [p3, p4] = p
                    m13 = (p1, p3, simpleDistance p1 p3)
                    m14 = (p1, p4, simpleDistance p1 p4)
                    m23 = (p2, p3, simpleDistance p2 p3)
                    m24 = (p2, p4, simpleDistance p2 p4)

edges e (p:ps) = m13:(m14:(m23:(m24:(edges e ps))))
                 where [p1, p2] = e
                       [p3, p4] = p
                       m13 = (p1, p3, simpleDistance p1 p3)
                       m14 = (p1, p4, simpleDistance p1 p4)
                       m23 = (p2, p3, simpleDistance p2 p3)
                       m24 = (p2, p4, simpleDistance p2 p4)

allMissingEdges [] = []

allMissingEdges [p] = []

allMissingEdges (e:es) =
   (edges e es) ++ (allMissingEdges es)

piecesFaults [] = []

piecesFaults (t:ts) = ps:(piecesFaults ts)
                      where (_, ps) = t

missingEdges tour = buildPQ (allMissingEdges (piecesFaults tour))

--------------------------------------------------------------------------------
-- Faz o merge de varios pedacos

data ConnectPiece a =
   Discard
 | ConnectOK(a)
 deriving (Eq, Read, Show)

connectPiece (j, k) [] = error "\nErro PQgreedy.088 -> O Caixeiro Viajante se perdeu"

connectPiece (j, k) [t] = Discard

connectPiece (j, k) tour@(p1:(p2:ps))
   | ((j == i1) && (k == i2)) ||
     ((k == i1) && (j == i2)) = ConnectOK(((reverse a) ++ (i1:(i2:b)), [f1, f2]):ps)
   | ((j == i1) && (k == f2)) ||
     ((k == i1) && (j == f2)) = ConnectOK((b ++ (f2:(i1:a)), [i2, f1]):ps)
   | ((j == f1) && (k == i2)) ||
     ((k == f1) && (j == i2)) = ConnectOK((a ++ (f1:(i2:b)), [i1, f2]):ps)
   | ((j == f1) && (k == f2)) ||
     ((k == f1) && (j == f2)) = ConnectOK((a ++ (f1:(f2:(reverse b))), [i1, i2]):ps)
   | otherwise = case (connectPiece (j, k) (p2:ps)) of
                 ConnectOK(newTour) -> ConnectOK(p1:newTour)
                 Discard -> Discard
   where (a, [i1, f1]) = p1
         (b, [i2, f2]) = p2

mergeTour LeafPQ tour _ = error "\nErro PQgreedy.107 -> O Caixeiro Viajante se perdeu"

mergeTour _ tour 1 = tour

mergeTour pq tour lim =
   case (connectPiece (j, k) tour) of
   ConnectOK(newTour) -> mergeTour (delMinPQ pq) newTour (lim - 1)
   Discard -> mergeTour (delMinPQ pq) tour lim
   where (j, k, _) = minPQ pq

--------------------------------------------------------------------------------
-- Tenta conectar um pedaco de tour a um dos outros pecacos do tour

data Insertion a =
   NewPiece([([Point a], [Point a])])
 | UpdateTree1(Point a, [([Point a], [Point a])])
 | UpdateTree2(Point a, Point a, [([Point a], [Point a])])
 | FindOtherNear(Point a, Point a)
 deriving (Eq, Read, Show)

merge c piece [] = UpdateTree1(c, [piece])

merge c piece@(a, [i1, f1]) (p:ps)
   | i1 == i2 = UpdateTree2(c, i1, ((reverse a) ++ (i2:b), [f1, f2]):ps)
   | i1 == f2 = UpdateTree2(c, i1, (b ++ (i1:a), [i2, f1]):ps)
   | f1 == i2 = UpdateTree2(c, f1, (a ++ (i2:b), [i1, f2]):ps)
   | f1 == f2 = UpdateTree2(c, f1, (a ++ (f2:(reverse b)), [i1, i2]):ps)
   | otherwise = case (merge c piece ps) of
                 UpdateTree1(a, newTour) -> UpdateTree1(a, p:newTour)
                 UpdateTree2(a, b, newTour) -> UpdateTree2(a, b, p:newTour)
   where (b, [i2, f2]) = p

--------------------------------------------------------------------------------
-- Tenta inserir uma aresta num tour

insertEdge (c, cl) [] = NewPiece([([], [c, cl])])

insertEdge (c, cl) tour@(p:ps)
   | c == i =  if cl /= f
               then merge c (i:es, [cl, f]) ps
               else FindOtherNear(c, cl)
   | c == f =  if cl /= i
               then merge c (es ++ [f], [i, cl]) ps
               else FindOtherNear(c, cl)
   | cl == i = merge cl (i:es, [c, f]) ps
   | cl == f = merge cl (es ++ [f], [i, c]) ps
   | otherwise = case (insertEdge (c, cl) ps) of
                 NewPiece(newTour) -> NewPiece(p:newTour)
                 UpdateTree1(a, newTour) -> UpdateTree1(a, p:newTour)
                 UpdateTree2(a, b, newTour) -> UpdateTree2(a, b, p:newTour)
                 FindOtherNear(a, b) -> FindOtherNear(a, b)
   where (es, [i, f]) = p

--------------------------------------------------------------------------------
-- Calcula o tamanho de um tour

pieceLength (ps, _) = 2 + (length ps)

tourLength [] = 0

tourLength (t:ts) = (pieceLength t) + (tourLength ts)

--------------------------------------------------------------------------------
-- Implementacao  da heuristica Greedy

pqGreedyAux _ LeafPQ tour lim =
   if (tourLength tour) == lim
   then mergeTour (missingEdges tour) tour (length tour)
   else error "\nErro PQgreedy.175 -> O Caixeiro Viajante se perdeu"

pqGreedyAux tree pq tour lim
   | (tourLength tour) > lim =
        error "\nErro Greedy.179 -> O Caixeiro Viajante se perdeu"
   | (tourLength tour) == lim =
        mergeTour (missingEdges tour) tour (length tour)
   | (not (isPresent p1 tree)) && (not (isPresent p2 tree)) =
        pqGreedyAux tree (delMinPQ pq) tour lim
   | (not (isPresent p1 tree)) =
        pqGreedyAux tree (insertPQ (delMinPQ pq) (nearSearchWE p2 tree)) tour lim
   | (not (isPresent p2 tree)) =
        pqGreedyAux tree (insertPQ (delMinPQ pq) (nearSearchWE p1 tree)) tour lim
   | otherwise = case (insertEdge (p1, p2) tour) of
                 NewPiece(newTour) ->
                    pqGreedyAux tree (delMinPQ pq) newTour lim
                 UpdateTree1(a, newTour) ->
                    pqGreedyAux (delete a tree) (delMinPQ pq) newTour lim
                 UpdateTree2(a, b, newTour) ->
                    pqGreedyAux (delete b (delete a tree)) (delMinPQ pq) newTour lim
                 FindOtherNear(a, b) ->
                    pqGreedyAux tree (insertPQ (delMinPQ pq) (nearSearchWE a (delete b tree))) tour lim
   where (p1, p2, _) = minPQ pq

pqGreedy [] = error "\nErro PQgreedy.199 -> O Caixeiro Viajante nao tem para onde ir"

pqGreedy [x] = [x]

pqGreedy [x, y] = [x, y]

pqGreedy [x, y, z] = [x, y, z]

pqGreedy ps = a ++ [f, i]
              where tree = build ps
                    pq = buildPQ (allEdges ps tree)
                    [(a, [i, f])] = pqGreedyAux tree pq [] (length ps)
